# @copyright (c) 2002-2013 Acronis International GmbH. All rights reserved.

import importlib
import operator
import ast


#public domain "unrepr" implementation, found on the web and then improved.

class _Builder:
    def build(self, o):
        cls_name = o.__class__.__name__.lower()
        try:
            return getattr(self, 'build_' + cls_name)(o)
        except AttributeError as e:
            raise TypeError('parser does not recognize {0!r}'.format(cls_name)) from e

    @staticmethod
    def ast_node(s):
        """Return a Python3 ast Node compiled from a string."""
        p = ast.parse('__tempvalue__ = ' + s)
        return p.body[0].value

    def build_subscript(self, o):
        return self.build(o.value)[self.build(o.slice)]

    def build_index(self, o):
        return self.build(o.value)

    def build_call(self, o):
        callee = self.build(o.func)

        if o.args is None:
            args = list()
        else:
            args = [self.build(a) for a in o.args]

        if o.starargs:
            args.extend(self.build(o.starargs))

        if o.kwargs is None:
            kwargs = {}
        else:
            kwargs = self.build(o.kwargs)

        if o.keywords:
            for kw in (self.build(x) for x in o.keywords):
                kwargs.update(kw)
        return callee(*args, **kwargs)

    def build_list(self, o):
        return [self.build(e) for e in o.elts]

    @staticmethod
    def build_str(o):
        return o.s

    @staticmethod
    def build_num(o):
        return o.n

    @staticmethod
    def build_bytes(o):
        return o.s

    def build_dict(self, o):
        return {self.build(k): self.build(v) for k, v in zip(o.keys, o.values)}

    def build_keyword(self, o):
        return {o.arg: self.build(o.value)}

    def build_tuple(self, o):
        return tuple(self.build(e) for e in o.elts)

    def build_set(self, o):
        return {self.build(e) for e in o.elts}

    @staticmethod
    def build_name(o):
        name = o.id
        if name == 'None':
            return None
        if name == 'True':
            return True
        if name == 'False':
            return False

        # See if the Name is a package or module. If it is, import it.
        try:
            return importlib.import_module(name)
        except ImportError:
            pass

        # See if the Name is in builtins.
        try:
            import builtins
            return getattr(builtins, name)
        except AttributeError:
            pass

        raise TypeError('unrepr could not resolve the name {0!r}'.format(name))

    def build_unaryop(self, o):
        op, operand = map(self.build, [o.op, o.operand])
        return op(operand)

    def build_binop(self, o):
        left, op, right = map(self.build, [o.left, o.op, o.right])
        return op(left, right)

    def build_compare(self, o):
        left, *comparators = map(self.build, [o.left] + o.comparators)
        return operator.is_(left, *comparators)

    @staticmethod
    def build_add(_):
        return operator.add

    @staticmethod
    def build_mult(_):
        return operator.mul

    @staticmethod
    def build_div(_):
        return operator.truediv

    @staticmethod
    def build_sub(_):
        return operator.sub

    @staticmethod
    def build_not(_):
        return operator.not_

    def build_attribute(self, o):
        parent = self.build(o.value)
        return getattr(parent, o.attr)

    @staticmethod
    def build_nonetype(_):
        return None

_builder = _Builder()


def unrepr(r):
    """Return a Python object compiled from a string."""
    if not isinstance(r, str):
        return r
    obj = _builder.ast_node(r)
    return _builder.build(obj)
