import acrort

from acrobind import get_dml_template, construct_from_template
from . constants import *
from . request_tokens import *


STAGE_MAPPING = {
    ARG_ARCHIVE_NAME: '.ArchiveName',
    ARG_LOCATION_URI: '.LocationUri',
    ARG_LOCATION_URI_TYPE: ('.LocationUriType', acrort.plain.DWORD),
    ARG_LOCATION_TYPE: ('.DestinationKind', acrort.plain.DWORD),
    ARG_LOCATION_CRED: '.LocationCredentials',
    ARG_STAGE_WITH_PLAN_CRED: ('.UseProtectionPlanCredentials', acrort.plain.BOOL)
}


def pred(*args):
    return acrort.plain.make_predicate_from_properties(args)


def format_error_no_property(property_name):
    return "Mandatory property '{}' not found in request for protection plan construction.".format(property_name)


def format_error_property_empty_string(property_name):
    return "Property '{}' must be a not empty string.".format(property_name)


def format_error_property_not_a_string(property_name):
    return "Property '{}' must be a string.".format(property_name)


def format_error_property_not_an_array(property_name):
    return "Property '{}' must be an array.".format(property_name)


def format_error_property_not_a_guid(property_name):
    return "Property '{}' must be a guid.".format(property_name)


def format_error_property_empty_array(property_name):
    return "Property '{}' must be a not empty array.".format(property_name)


def format_error_no_defaults_for_scheme(sheme_type):
    return "Backup scheme '{}' construction from default parameters is not supported.".format(SCHEME_TITLES.get(sheme_type))


def format_error_poly_type_missing(class_info):
    return "'polyType' field which defines subclass of '{}' is missing".format(class_info)


def format_error_poly_type_wrong(poly_type, class_info):
    return "'polyType' field which defines subclass of '{}' has wrong value {}".format(class_info, poly_type)


def format_error_root_field_missing(field):
    return "root field {} is missing".format(field)


class RequestReader:
    def __init__(self, request):
        if not isinstance(request, acrort.plain.Unit):
            self._request = acrort.plain.Unit(request)
        else:
            self._request = request

    def _get_guid(self, prop_name):
        unit = self._request.get(prop_name, None)
        if unit is None:
            acrort.common.make_logic_error(format_error_no_property(prop_name)).throw()
        if not unit.is_data() or (unit.type != acrort.plain.GUID):
            acrort.common.make_logic_error(format_error_property_not_a_guid(prop_name)).throw()
        return unit.ref

    def _get_array_by_path(self, prop_path):
        unit = self._request.get_branch(prop_path, None)
        if unit is None:
            acrort.common.make_logic_error(format_error_no_property(prop_path)).throw()
        if not unit.is_array():
            acrort.common.make_logic_error(format_error_property_not_an_array(prop_path)).throw()
        return unit

    def _get_not_empty_array_by_path(self, prop_path):
        unit = self._get_array_by_path(prop_path)
        if len(unit) == 0:
            acrort.common.make_logic_error(format_error_property_empty_array(prop_path)).throw()
        return unit

    def _get_not_empty_string(self, prop_name):
        unit = self._request.get(prop_name, None)
        if unit is None:
            acrort.common.make_logic_error(format_error_no_property(prop_name)).throw()
        if not unit.is_data() or (unit.type != acrort.plain.STRING):
            acrort.common.make_logic_error(format_error_property_not_a_string(prop_name)).throw()
        if not unit.ref:
            acrort.common.make_logic_error(format_error_property_empty_string(prop_name)).throw()
        return unit.ref

    def get_plan_id(self):
        return self._get_guid(ARG_PLAN_ID)

    def get_plan_name(self):
        return self._get_not_empty_string(ARG_PLAN_NAME)

    def get_backup_type(self):
        return self._get_not_empty_string(ARG_BACKUP_TYPE)

    def get_inclusions(self):
        inclusions = self._get_array_by_path(pred(ARG_TARGET, ARG_INCLUSIONS))
        result = []
        for idx, item in inclusions:
            key = inclusions[idx][ARG_INCLUSION_KEY]
            options = inclusions[idx].get(ARG_INCLUSION_OPTIONS)
            result.append((key, options))
        return result

    def get_scheme_type(self):
        unit = self._request.get(ARG_SCHEME, {}).get(ARG_TYPE)
        if unit is not None:
            return unit.ref

    def get_scheme_parameters(self):
        return self._request.get(ARG_SCHEME, {}).get(ARG_PARAMETERS)

    def get_stages(self):
        return self._get_not_empty_array_by_path(pred(ARG_ROUTE, ARG_STAGES))

    def get_archive_slicing(self):
        return self._request.get(ARG_ROUTE, {}).get(ARG_ARCHIVE_SLICING)

    def get_options(self):
        return self._request.get(ARG_OPTIONS)

    def get_tenant(self):
        return self._request.get(ARG_TENANT)


def construct_protection_set(inclusions, get_template):
    inclusion_template = get_template(TAG_GTOB_DTO_PROTECTION_SOURCE)
    protection_set_template = get_template(TAG_GTOB_DTO_PROTECTION_SET)
    sources = []
    for key, options in inclusions:
        patch = [('.Key', key)]
        if options:
            patch.append(('.Options', options))
        item = inclusion_template.merge(acrort.plain.UnitDiff(patch))
        sources.append(item)
    protection_set = protection_set_template.merge(acrort.plain.UnitDiff([
        ('.Inclusions', acrort.plain.Unit(sources, traits=protection_set_template.Inclusions.traits))
    ]))
    return protection_set


def construct_protection_stages(stages, get_template):
    stage_template = None
    staging_rule_template = None
    result = []
    for idx, stage in stages:
        if stage.is_instance(TAG_GTOB_DTO_STAGE):
            result.append(stage)
            continue
        if stage_template is None:
            stage_template = get_template(TAG_GTOB_DTO_STAGE)
        patch = []
        for name, patch_info in STAGE_MAPPING.items():
            value = stage.get(name)

            if value is None:
                continue
            if value.is_data():
                value = value.ref

            patch_placement = None
            flat_type = None

            if isinstance(patch_info, tuple):
                patch_placement = patch_info[0]
                flat_type = patch_info[1]
            else:
                patch_placement = patch_info

            branch_template = stage_template.get_branch(patch_placement)

            if flat_type is None:
                patch_subject = acrort.plain.Unit(value, traits=branch_template.traits)
            else:
                patch_subject = acrort.plain.Unit(flat_type, value, traits=branch_template.traits)

            patch_entry = (patch_placement, patch_subject)
            patch.append(patch_entry)

        configuration = stage.get('Configuration')
        if configuration:
            rules = configuration.get('StagingRulesSet')
            if rules:
                if not staging_rule_template:
                    staging_rule_template = get_template(TAG_GTOB_DTO_STAGING_RULES)

                staging_rules = []

                for idx, rule in rules:

                    alarms = [construct_alarm(alarm, get_template) for idx, alarm in rule['RetentionSchedule']['Alarms']]

                    alarms_patch_placement = '.RetentionSchedule.Alarms'
                    alarms_template = staging_rule_template.get_branch(alarms_patch_placement)
                    alarms_patch_subject = acrort.plain.Unit(alarms, traits=alarms_template.traits)
                    alarms_patch_entry = (alarms_patch_placement, alarms_patch_subject)

                    conditions = [construct_condition(cond, get_template) for idx, cond in rule['RetentionSchedule']['Conditions']]

                    conditions_patch_placement = '.RetentionSchedule.Conditions'
                    conditions_template = staging_rule_template.get_branch(conditions_patch_placement)
                    conditions_patch_subject = acrort.plain.Unit(conditions, traits=conditions_template.traits)
                    conditions_patch_entry = (conditions_patch_placement, conditions_patch_subject)

                    staging_rule = construct_from_template(staging_rule_template, rule)
                    staging_patches = [alarms_patch_entry, conditions_patch_entry]
                    staging_rule = staging_rule.merge(acrort.plain.UnitDiff(staging_patches))
                    staging_rules.append(staging_rule)

                staging_rules_patch_placement = '.Configuration.StagingRulesSet'
                staging_rules_template = stage_template.get_branch(staging_rules_patch_placement)

                patch_subject = acrort.plain.Unit(staging_rules, traits=staging_rules_template.traits)
                patch_entry = (staging_rules_patch_placement, patch_subject)
                patch.append(patch_entry)

        stage_unit = stage_template.merge(acrort.plain.UnitDiff(patch))
        result.append(stage_unit)

    return result


def construct_custom_polymorph_template(unit, name, classes, get_template):
    if name not in unit:
        acrort.common.make_logic_error(format_error_root_field_missing(name)).throw()
    unit = unit[name]
    if ARG_POLY_TYPE not in unit:
        acrort.common.make_logic_error(format_error_poly_type_missing(name)).throw()
    poly_type = unit[ARG_POLY_TYPE].ref
    if poly_type not in classes:
        acrort.common.make_logic_error(format_error_poly_type_wrong(poly_type, name)).throw()
    return acrort.plain.Unit({name: {ARG_POLY_TYPE: (acrort.plain.DWORD, 0),
                                     ARG_IMPL: get_template(classes[poly_type])}})


def construct_calendar(calendar, get_template):
    template = construct_custom_polymorph_template(calendar, ARG_CALENDAR, CALENDAR_CLASS_INFO, get_template)
    return construct_from_template(template, calendar)


def construct_alarm(alarm, get_template):
    template = construct_custom_polymorph_template(alarm, ARG_ALARM, ALARM_CLASS_INFO, get_template)
    template = template.merge(acrort.plain.UnitDiff(
        [(pred(ARG_ALARM, ARG_ONCE_A_DAY), acrort.plain.Unit(False))]))
    time_alarm_calendar = alarm.get_branch(pred(ARG_ALARM, ARG_IMPL, ARG_CALENDAR), None)
    if time_alarm_calendar is not None:
        cal_templ = construct_custom_polymorph_template(time_alarm_calendar, ARG_CALENDAR, CALENDAR_CLASS_INFO, get_template)
        template = template.merge(acrort.plain.UnitDiff([(pred(ARG_ALARM, ARG_IMPL, ARG_CALENDAR), cal_templ)]))
    return construct_from_template(template, alarm)


def construct_condition(condition, get_template):
    template = construct_custom_polymorph_template(condition, ARG_CONDITION, CONDITION_CLASS_INFO, get_template)
    return construct_from_template(template, condition)


def scheme_parameters_template(scheme_type, get_template):
    if scheme_type not in SCHEME_PARAMETERS_CLASS_INFO:
        acrort.common.make_logic_error(format_error_no_defaults_for_scheme(scheme_type)).throw()
    class_id = SCHEME_PARAMETERS_CLASS_INFO[scheme_type]
    return get_template(class_id)


def construct_backup_now_scheme_params(params, get_template):
    template = scheme_parameters_template(BACKUP_SCHEME_NOW, get_template)
    return construct_from_template(template, params)


def construct_run_once_scheme_params(params, get_template):
    template = scheme_parameters_template(BACKUP_SCHEME_ONCE, get_template)
    if params and ARG_BACKUP_TIME in params:
        backup_time_templ = get_template(TAG_GTOB_DTO_DATE_TIME)
        params = params.merge(acrort.plain.UnitDiff([
            ('.' + ARG_BACKUP_TIME, construct_from_template(backup_time_templ, params[ARG_BACKUP_TIME]))]))
    return construct_from_template(template, params)


def setup_schedule(unit, schedule_path, get_template):
    alarm_path = schedule_path + '.' + ARG_ALARMS
    condition_path = schedule_path + '.' + ARG_CONDITIONS
    alarms = [construct_alarm(alarm, get_template) for ind, alarm in unit.get_branch(alarm_path, [])]
    conditions = [construct_condition(cond, get_template) for ind, cond in unit.get_branch(condition_path, [])]
    patch = [(alarm_path, alarms), (condition_path, conditions)]
    return unit.merge(acrort.plain.UnitDiff(patch))


def construct_simple_scheme_params(params, get_template):
    template = scheme_parameters_template(BACKUP_SCHEME_SIMPLE, get_template)
    if params:
        params = setup_schedule(params, pred(ARG_BACKUP_SCHEDULE, ARG_SCHEDULE), get_template)
    return construct_from_template(template, params)


def construct_gfs_scheme_params(params, get_template):
    template = scheme_parameters_template(BACKUP_SCHEME_GFS, get_template)
    return construct_from_template(template, params)


def construct_toh_scheme_params(params, get_template):
    template = scheme_parameters_template(BACKUP_SCHEME_TOWER_OF_HANOI, get_template)
    if params and ARG_BASIC_ALARM in params:
        template = template.merge(acrort.plain.UnitDiff([(pred(ARG_BASIC_ALARM), 0)]))
        alarm = construct_alarm(params[ARG_BASIC_ALARM], get_template)
        params = params.merge(acrort.plain.UnitDiff([(pred(ARG_BASIC_ALARM), alarm)]))
    return construct_from_template(template, params)


def construct_backup_schedule_item(item, get_template):
    template = get_template(TAG_GTOB_DTO_BACKUP_SCHEDULE_ITEM)
    if item:
        item = setup_schedule(item, pred(ARG_SCHEDULE), get_template)
    return construct_from_template(template, item)


def construct_custom_scheme_params(params, get_template):
    template = scheme_parameters_template(BACKUP_SCHEME_CUSTOM, get_template)
    if params:
        items = [construct_backup_schedule_item(item, get_template) for ind, item in params.get_branch(pred(ARG_ITEMS), [])]
        params = params.merge(acrort.plain.UnitDiff([(pred(ARG_ITEMS), items)]))
    return construct_from_template(template, params)


def construct_specific_parameters(backup_type, get_template):
    specific_parameters = {'BackupType': backup_type}

    specific_parameters_template = get_template(TAG_GTOB_DTO_SPECIFIC_PROTECTION_PARAMETERS)

    specific_parameters_patch_subject = acrort.plain.Unit(specific_parameters, traits=specific_parameters_template.traits)

    return construct_from_template(specific_parameters_template, specific_parameters_patch_subject)


_SCHEME_PARAMS_CONSTRUCTORS = {
    BACKUP_SCHEME_NOW: construct_backup_now_scheme_params,
    BACKUP_SCHEME_ONCE: construct_run_once_scheme_params,
    BACKUP_SCHEME_SIMPLE: construct_simple_scheme_params,
    BACKUP_SCHEME_GFS: construct_gfs_scheme_params,
    BACKUP_SCHEME_TOWER_OF_HANOI: construct_toh_scheme_params,
    BACKUP_SCHEME_CUSTOM: construct_custom_scheme_params,
    BACKUP_SCHEME_ALWAYS_FULL: construct_simple_scheme_params,
    BACKUP_SCHEME_ALWAYS_INCREMENTAL: construct_simple_scheme_params,
    BACKUP_SCHEME_WEEKLY_FULL_DAILY_INCR: construct_simple_scheme_params,
    BACKUP_SCHEME_MONTHLY_FULL_WEEKLY_DIFF_DAILY_INCR: construct_gfs_scheme_params
}


def construct_plan(request, connection):
    template_cache = {}

    def get_template(type_name):
        template = template_cache.get(type_name)
        if template is None:
            template = get_dml_template(type_name, connection)
            template_cache[type_name] = template
        return template

    reader = RequestReader(request)
    plan_id = reader.get_plan_id()
    plan_name = reader.get_plan_name()
    backup_type = reader.get_backup_type()
    inclusions = reader.get_inclusions()
    stages = reader.get_stages()
    archive_slicing = reader.get_archive_slicing()
    scheme_type = reader.get_scheme_type()
    scheme_parameters = reader.get_scheme_parameters()
    tenant = reader.get_tenant()

    if scheme_type is None:
        scheme_type = BACKUP_SCHEME_ONCE

    scheme_params_constructor = _SCHEME_PARAMS_CONSTRUCTORS[scheme_type]

    template = get_template(TAG_GTOB_DTO_PROTECTION_PLAN)

    patch = [
        ('.ID', acrort.plain.Unit(plan_id, traits=template.ID.traits)),
        ('.PlanID', acrort.plain.Unit(plan_id, traits=[('PrimaryKey', None), ('TemplateSource', 'Plan ID')])),
        ('.OwnerID', ('guid', connection.user_profile_id)),
        ('.Name', acrort.plain.Unit(plan_name, traits=template.Name.traits)),
        ('.Target', construct_protection_set(inclusions, get_template)),
        ('.Scheme.Type', ('dword', scheme_type)),
        ('.Scheme.Parameters', scheme_params_constructor(scheme_parameters, get_template)),
        ('.Route.Stages', acrort.plain.Unit(construct_protection_stages(stages, get_template), traits=template.Route.Stages.traits)),
        ('.Route.ArchiveSlicing', acrort.plain.Unit(archive_slicing))
    ]

    if backup_type not in [BACKUP_TYPE_SQL, BACKUP_TYPE_EXCHANGE, BACKUP_TYPE_SYSTEM_STATE]:
        patch += [('.Options.SpecificParameters', construct_specific_parameters(backup_type, get_template))]

    if tenant is not None:
        patch += [('.Tenant', acrort.plain.Unit(tenant))]

    plan = template.merge(acrort.plain.UnitDiff(patch))

    options = reader.get_options()
    if options is not None:
        plan = construct_from_template(plan, acrort.plain.Unit({ARG_OPTIONS: options}))

    return plan
