# @copyright (c) 2002-2013 Acronis International GmbH. All rights reserved.

from acronis.lib.algorithms import for_each
from email import encoders
from email.mime.base import MIMEBase
from email.mime.text import MIMEText
from email.header import Header
from email.mime.multipart import MIMEMultipart
import smtplib
import ssl
import os
import socket


SMTPException = smtplib.SMTPException


def _idn_to_ascii(s):
    return s.encode('idna').decode('ascii')

def _idn_to_ascii_email(s):
    return '@'.join(map(_idn_to_ascii, s.split('@')))

def _make_attachment(file):
    if isinstance(file, dict):
        path = file['path']
        alias = file['alias']
        ctype = file.get('type', 'application/octet-stream')
    else:
        path = file
        alias = os.path.basename(file)
        ctype = 'application/octet-stream'

    main_type, _, sub_type = ctype.partition('/')
    attachment = MIMEBase(main_type, sub_type)
    with open(path, 'rb') as stream:
        attachment.set_payload(stream.read())
    encoders.encode_base64(attachment)
    attachment.add_header('Content-Disposition', 'attachment', filename=(Header(alias, 'utf-8').encode()))
    return attachment


class Message(MIMEMultipart):
    def __init__(self, sender, subject, message, recipients, files=None, bodytype='plain', hidden_recipients=None, _subtype='mixed'):
        super().__init__(_subtype=_subtype)
        self.attach(MIMEText(message, bodytype))
        self['Subject'] = subject or ''
        self['From'] = sender or ''
        self['Sender'] = sender or ''
        self['To'] = ','.join(map(_idn_to_ascii_email, recipients)) if recipients else None
        self['Bcc'] = ','.join(map(_idn_to_ascii_email, hidden_recipients)) if hidden_recipients else None
        if files is not None:
            for_each(files, lambda x: self.attach(_make_attachment(x)))


class SmtpClient:
    def __init__(self, host=None, port=0, username=None, password=None, encryption_algo=None, timeout=None):
        encryption_algo = encryption_algo.upper() if encryption_algo else ''
        smtp_class = smtplib.SMTP_SSL if encryption_algo in ('SSL',) else smtplib.SMTP
        encryption_algo = encryption_algo.upper()
        hostname = socket.getfqdn()
        self.smtp = smtp_class(host=host or '127.0.0.1', port=port, timeout=timeout or 5, local_hostname=_idn_to_ascii(hostname))
        if encryption_algo == 'TLS':
            # Create a secure SSL context with certificate validation
            context = ssl.create_default_context()
            self.smtp.starttls(context=context)
        if username and password:
            self.smtp.login(username, password)

    def __enter__(self):
        return self.smtp

    def __exit__(self, *_):
        try:
            self.smtp.quit()
        except smtplib.SMTPServerDisconnected:
            pass
        return False


class SmtpConnection:
    def __init__(self, **kwargs):
        self.arguments = kwargs

    def connect(self):
        return SmtpClient(**self.arguments)


class OutboxError(SMTPException):
    def __init__(self, message):
        self.message = message

    def __str__(self):
        return 'Subject: {subj}; To: {to}; Cc: {cc}; Bcc: {bcc}'.format(
            subj=self.message.get('Subject', ''), to=self.message.get('To', ''),
            cc=self.message.get('Cc', ''), bcc=self.message.get('Bcc', '')
        )


class Outbox:
    def __init__(self, connection=None, try_count=5):
        self.connection = connection or SmtpConnection()
        self.try_count = try_count

    def send(self, message):
        attempt = 1
        while True:
            try:
                with self.connection.connect() as connection:
                    try:
                        connection.send_message(message)
                    finally:
                        # Don't retry sending - only connecting
                        break
            except smtplib.SMTPServerDisconnected:
                if attempt >= self.try_count:
                    raise OutboxError(message)
                attempt += 1
