import acrort

from . cmdline_parser import ArgumentsProcessor, ArgumentParser


TAG_CONNECTION = 'connection'
TAG_SERVICE = 'service'
TAG_PIPE = 'pipe'
TAG_HOSTNAME = 'hostname'
TAG_USERNAME = 'username'
TAG_PASSWORD = 'password'
TAG_TENANT = 'tenant'
TAG_TENANCY_REQUIRED = 'tenancy_required'


class ConnectionArgumentsProcessor(ArgumentsProcessor):
    def __init__(self, *, tenancy_required=False):
        self._tenancy_required = tenancy_required

    def create_arguments_parser(self):
        parser = ArgumentParser(add_help=False)
        parser.add_argument('-s', '--service', nargs=1, choices=acrort.connectivity.SUPPORTED_PROTOCOLS, required=True)
        connection_parser = parser.add_mutually_exclusive_group();
        connection_parser.add_argument('--connection', nargs=3, metavar=('HOST', 'USERNAME', 'PASSWORD'))
        connection_parser.add_argument('--pipe', metavar='LOCALPIPE')
        parser.add_argument('--tenant', metavar='TENANT')

        return parser

    def process_arguments(self, config, args):
        output = {}
        output[TAG_SERVICE] = args.service[0]
        config[TAG_TENANCY_REQUIRED] = self._tenancy_required
        if args.connection is not None:
            output[TAG_HOSTNAME] = args.connection[0]
            output[TAG_USERNAME] = args.connection[1]
            output[TAG_PASSWORD] = args.connection[2]
        elif args.pipe is not None:
            config[TAG_PIPE] = args.pipe
        config[TAG_CONNECTION] = output
        if args.tenant is not None:
            config[TAG_TENANT] = args.tenant


def connect_to_service(config, *, queue=None, printer=None):
    conn_config = config[TAG_CONNECTION]
    service = conn_config[TAG_SERVICE]
    pipename = None
    username = None
    tenancy_required = False
    if TAG_TENANCY_REQUIRED in config:
        tenancy_required = config[TAG_TENANCY_REQUIRED]
    if TAG_HOSTNAME in conn_config:
        hostname = conn_config[TAG_HOSTNAME]
        username = conn_config[TAG_USERNAME]
        password = conn_config[TAG_PASSWORD]
        if printer is not None:
            printer.write("Connecting to {} on {} ...".format(service, hostname), end='\n')
        args = (service, hostname, (username, password))
    elif TAG_PIPE in config:
        pipename = config[TAG_PIPE]
        if printer is not None:
            printer.write("Connecting locally to {} by {} ...".format(service, pipename), end='\n')
        args = (service,)
    else:
        if printer is not None:
            printer.write("Connecting locally to {} ...".format(service), end='\n')
        args = (service,)
    tenant = None
    if TAG_TENANT in config:
        tenant = config[TAG_TENANT]
    client_session_data = {}
    if tenant is not None:
        client_session_data["tenant_id"] = tenant
    elif tenancy_required:
        client_session_data["identity_disabled"] = True
    try:
        connection = acrort.connectivity.Connection(*args, pipe=pipename, client_session_data=client_session_data, queue=queue)
    except acrort.Exception as exception:
        error = exception.to_error()
        if printer is not None:
            printer.write(error, end='\n')
        return (None, error)
    if printer is not None:
        printer.write('Connected to {} (version: {}) as {}'.format(connection.service_name, connection.service_version, connection.user_name), end='\n')
        tenant_id = None
        if isinstance(connection.properties, dict):
            tenant_id = connection.properties.get('Tenant', {}).get('ID')
        if tenant_id is not None:
            printer.write('Connected to {} as tenant {}'.format(connection.service_name, tenant_id), end='\n')
    return (connection, acrort.common.SUCCESS)
