from . constants import *
from . import logs
import datetime
import urllib.parse

import acrort
import sys
import logging


def epoch_to_time(e):
    return datetime.datetime.fromtimestamp(e, datetime.timezone.utc).astimezone()


def get_slice_id(slice):
    for _, pair in slice.MetaInformation:
        if pair.first == 'slice-id':
            logging.info("[slices][get_slice_id] Slice (the oldest slice is shown first): {}".format(pair.second.ref))
            return pair.second.ref
    return None


def make_stage_spec(backup_plan, stage):
    return {
        'location_uri': stage.LocationUri,
        'location_credentials': stage.LocationCredentials,
        'archive_name': backup_plan.Route.Stages[0].ArchiveName,
        'archive_display_name': backup_plan.Route.Stages[0].ArchiveDisplayName,
        'rules': stage.Configuration.StagingRulesSet,
        'archive_slicing': backup_plan.Route.ArchiveSlicing,
        'destination_kind': stage.DestinationKind,
        'validation_options': stage.ValidationOptions,
        'task_execution_window': stage.Configuration.TaskExecutionWindow,
    }


def make_archive_spec(archive):
    result = {
        'uri': archive.ArchiveUri.ref,
        'size': archive.Size.ref,
    }
    for i, attribute in archive.Attributes:
        result[attribute["first"].ref] = attribute["second"].ref
    for i, attribute in archive.Settings.Attributes:
        result[attribute["first"].ref] = attribute["second"].ref
    return result


def make_slice_spec(slice):
    return {
        'kind': slice.Kind.ref,
        'creation_time': epoch_to_time(slice.CreationTime.ref),
        'id': get_slice_id(slice),
        'uri': slice.SliceUri.ref,
        'size': slice.Size.ref
    }


def select_slices(service, stage, backup_options, ignore_missing_archive, include_immutable=False):
    location_uri=stage['location_uri'].ref
    if not urllib.parse.urlparse(location_uri).fragment.startswith('arl:'):
        location_uri = '{0}#{1}'.format(
            location_uri,
            'arl:/00000000-0000-0000-0000-000000000000/00000000-0000-0000-0000-000000000000/00000000-0000-0000-0000-000000000000') 
    archive_uri = '{0}?{1}'.format(
        location_uri,
        # ABR-324484 : Common::Uri incorrectly works with uri's query, so we have to encode values twice
        urllib.parse.urlencode({'archive_name': urllib.parse.quote(stage['archive_name'].ref)}, quote_via=urllib.parse.quote)
    )

    tree = service.gtob.open_global_tree()
    archive_key = acrort.gtob.ItemKey(archive_uri, 'dms:archive')
    try:
        options = {}
        options.update({'AccessToken': stage['location_credentials']})
        options.update({'IncludeDeletedImmutable' : False })
        options.update({'IncludeImmutableArchive' : include_immutable })
        if backup_options['BackupOptions']['ArchiveProtection']['UsePassword'].ref:
            options.update({'ArchiveCredentialsId': backup_options['BackupOptions']['ArchiveProtection']['PasswordUrl']})

        archive = tree.resolve_item(key=archive_key, options=options)
        archive.throw_if_error_item()
        archive_key = acrort.gtob.ItemKey(archive.key.local_id, 'dms:archive')
        slices = tree.select_children(key=archive_key, options=options)
        for s in slices:
            s.throw_if_error_item()
        selected_slices = [make_slice_spec(slice.attributes) for slice in slices]
        # Do ignore CDP slices
        selected_slices = [s for s in selected_slices if s['kind'] != SLICE_KIND_CDP]
        logging.info("[slices][select_slices] Archive details: {}".format(make_archive_spec(archive.attributes)))
        logging.info("[slices][select_slices] Slice kind:\n0 = full\n1 = incremental\n2 - differential\n3 - NA\n4 - CDP")
        logging.info("[slices][select_slices] Selected slices: {}".format(selected_slices))
        return make_archive_spec(archive.attributes), selected_slices
    except acrort.Exception as e:
        error = e.to_error()
        failed_to_find_archive = error.find_suberror(0xa100d4) is not None
        cannot_find_archive = error.find_suberror(0xa10007) is not None
        unknown_archive = error.find_suberror(0x270030) is not None
        # ignore_missing_archive not checked for quick fix of ABR-70329
        if (failed_to_find_archive or cannot_find_archive or unknown_archive):
            logging.info("[slices][select_slices] No archive found to select slices from.")
            return None, None
        raise


def select_last_slice(service, stage, backup_options):
    _, slices = select_slices(service, stage, backup_options, ignore_missing_archive = True)
    if slices is None or len(slices) == 0:
        logging.info("[slices][select_last_slice] No slices inside the archive")
        return None
    logging.info("[slices][select_last_slice] Last created slice:\n{}".format(slices[-1]))
    return slices[-1]

# Sort slices by time in ascending order
def sort_by_creation_time(slices):
    return sorted(slices, key=lambda x: x['creation_time'], reverse=False)
